<?php

namespace routes;

use model\Levels;
use Api\BaseRouter;
use model\Startups;
use SocialMediaApi;
use FwRoutingSystem\Router;
use model\Entity\LevelsEntity;
use model\Entity\StartupsEntity;
use model\Entity\SocialMediaServicesEntity;

class SocialMediaRoute extends BaseRouter {
	public $groupPath = 'social-media';
	public $startup;
	
	public function __construct() {
		$token = get_header('token');
		if ($token) {
			$this->startup = Startups::getByToken($token);
		}
	}
	
	public function requiredHeaders() : array {
		return [
			'token' => function ($token) {
				return [
					'isCustom' => true,
					'validate' => Startups::getByToken($token) instanceof StartupsEntity,
					'message'  => [
						'message' => 'client not found',
						'hint'    => 'you can obtain a new token by using the {obtainToken} endpoint',
					],
				];
			},
		];
	}
	
	public function routes(Router $router) {
		$router->get('/services', $this->middleware($router), function () {
			return response(collect($this->apiClass()->services())->map(function ($service) {
				return $this->mapServiceFunction($service);
			}));
		});
		$router->get('/services/:id', $this->middleware($router), function ($id) {
			return response(collect($this->apiClass()->services())->filter(function ($item) use ($id) {
				return $item->service == "$id";
			})->map(function ($service) {
				return $this->mapServiceFunction($service);
			})->values());
		});
		$router->get('/services/:string', $this->middleware($router), function ($type) {
			return response(collect($this->apiClass()->services())->filter(function ($item) use ($type) {
				return $item->type == "$type";
			})->map(function ($service) {
				return $this->mapServiceFunction($service);
			})->values());
		});
		$router->get('/type-info/:string', $this->middleware($router), function ($type) {
			return response($this->apiClass()->typeInfo($type));
		});
		$router->get('/type-info/:id', $this->middleware($router), function ($id) {
			$item = collect($this->apiClass()->services())->filter(function ($item) use ($id) {
				return $item->service == "$id";
			})->map(function ($service) {
				return $this->mapServiceFunction($service);
			});
			return response($this->apiClass()->typeInfo($item->first->type));
		});
		$router->get('/categories', $this->middleware($router), function () {
			return response(collect($this->apiClass()->socialMedias())->map(function ($item, $key) {
				unset($item['filter']);
				$item['key'] = $key;
				return $item;
			})->values());
		});
		$router->get('/services-by-category/:string', $this->middleware($router), function ($category) {
			$categoryItem = $this->apiClass()->socialMedias();
			if (isset($categoryItem[$category])) {
				$categoryItem = $categoryItem[$category];
				$filterFunction = $categoryItem['filter'];
				$services = $this->apiClass()->services();
				return response(collect($services)->filter(function ($service) use ($filterFunction) {
					return $filterFunction($service);
				})->map(function ($service) {
					return $this->mapServiceFunction($service);
				})->values());
			} else {
				return response(['message' => 'category not found'], 404);
			}
		});
		$router->post('/order/:id', $this->middleware($router), function ($serviceId) {
			$singleService = collect($this->apiClass()->services())->filter(function ($item) use ($serviceId) {
				return $item->service == "$serviceId";
			})->values();
			if ($singleService->length() > 0) {
				$singleService = $singleService->first;
				$singleService = $this->mapServiceFunction($singleService);
				return response($this->apiClass()->addOrder($singleService, $_POST));
			}
			return response(['message' => 'service not found'], 404);
		});
	}
	
	/**
	 * @return SocialMediaApi
	 */
	public function apiClass() {
		return new SocialMediaApi();
	}
	
	private function mapServiceFunction($service) {
		$id = $service->service;
		$item = \model\SocialMediaServices::getOneFiltered('service_id', $id);
		$value = $item instanceof SocialMediaServicesEntity ? $item->media_service_percent : 30;
		$rate = (int)$service->rate;
		$rate = $rate + ($rate * $value / 100);
		$service->rate = (int)$rate;
		// change name
		$name = str($service->name);
		$name = $name->replace('نایس پنل', 'نگامدیا');
		$name = $name->replace('نایسپنل', 'نگامدیا');
		$name = $name->replace('nice panel', 'نگامدیا');
		$name = $name->replace('nicepanel', 'نگامدیا');
		$service->name = $name->getValue();
		// change category
		$category = str($service->category);
		$category = $category->replace('نایس پنل', 'نگامدیا');
		$category = $category->replace('نایسپنل', 'نگامدیا');
		$category = $category->replace('nice panel', 'نگامدیا');
		$category = $category->replace('nicepanel', 'نگامدیا');
		$service->category = $category->getValue();
		// change desc
		$desc = str($service->desc);
		$desc = $desc->replace('نایس پنل', 'نگامدیا');
		$desc = $desc->replace('نایسپنل', 'نگامدیا');
		$desc = $desc->replace('nice panel', 'نگامدیا');
		$desc = $desc->replace('nicepanel', 'نگامدیا');
		$service->desc = $desc->getValue();
		$levels = json_decode($item->levels, true);
		$service->levels = Levels::getAllFiltered('startup_id', $this->startup->id)->map(function (LevelsEntity $level) {
			return [
				'id'       => $level->level_id,
				'name'     => $level->title,
				'isActive' => true,
				'percent'  => 0,
			];
		});
		if ($levels) {
			if (isset($levels[$this->startup->id])) {
				$levelsForStartup = $levels[$this->startup->id];
				$service->levels->map(function ($level) use ($levelsForStartup) {
					$item = 0;
			
					if (isset($levelsForStartup[$level['id']])) {
						$item = $levelsForStartup[$level['id']];
					}
					return [
						'id'       => $level['id'],
						'name'     => $level['name'],
						'isActive' => $item >= 0,
						'percent'  => (int)$item,
					];
				});
			}
		}
		return $service;
	}
	
	
}
